/* 
 * Communications via the FrSky Data telemetry port.
 */

#include <SoftwareSerial.h>

#define FRSKY_MAGIC 0x5E
#define FRSKY_ESC 0x5D

// We're doing software serial so we can invert the logic. Silly FrSky.
static SoftwareSerial FrSky_data(0, 8, true);

bool FrSky_setup() {
#ifdef SERIAL_DEBUG
  Serial.println("Initializing data serial line.");
#endif
  FrSky_data.begin(9600);
  return true;
}

void FrSky_start_frame() {
  FrSky_data.write(FRSKY_MAGIC);
}

static void FrSky_send_byte(uint8_t data) {
#ifdef SERIAL_DEBUG
  Serial.print("Sending ");
  Serial.println(data);
#endif

  switch (data) {
  case FRSKY_MAGIC: 
  case FRSKY_ESC:
    FrSky_data.write(FRSKY_ESC);
    FrSky_data.write(data ^ 0x60);
    break;
  default:
    FrSky_data.write(data);
    break;
  }
}
    
void FrSky_send_data(uint8_t id, uint16_t data) {
  FrSky_send_byte(id);
  FrSky_send_byte(data & 0xff);
  FrSky_send_byte(data >> 8);
  FrSky_data.write(FRSKY_MAGIC);
}

void FrSky_send_float(uint8_t id, int16_t whole, uint16_t frac) {
  switch (id) {
  case 0x01: case 0x11: case 0x14: case 0x3B:
    FrSky_send_data(id, (uint16_t) whole) ;
    FrSky_send_data(id + 8, frac) ;
    break ;
  case 0x10:
    FrSky_send_data(id, whole);
    FrSky_send_data(0x21, frac);
    break;
  case 0x12:
    FrSky_send_data(id, abs(whole)) ;
    FrSky_send_data(0x1A, frac) ;
    FrSky_send_data(0x22, whole < 0 ? 'W' : 'E');
    break ;
  case 0x13:
    FrSky_send_data(id, abs(whole)) ;
    FrSky_send_data(0x1B, frac) ;
    FrSky_send_data(0x23, whole < 0 ? 'S' : 'N');
    break ;
  default:
#ifdef SERIAL_DEBUG
    Serial.print(F("FrSky: Unrecognized floating point id: "));
    Serial.println(id);
#endif
    break;
  }
}
